package net.minecraft.block;

import net.lax1dude.eaglercraft.EaglercraftRandom;
import javax.annotation.Nullable;
import net.minecraft.block.material.Material;
import net.minecraft.block.properties.IProperty;
import net.minecraft.block.properties.PropertyEnum;
import net.minecraft.block.properties.PropertyInteger;
import net.minecraft.block.state.BlockFaceShape;
import net.minecraft.block.state.BlockStateContainer;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.init.Blocks;
import net.minecraft.init.Items;
import net.minecraft.item.Item;
import net.minecraft.item.ItemBlock;
import net.minecraft.item.ItemStack;
import net.minecraft.stats.StatList;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.tileentity.TileEntityFlowerPot;
import net.minecraft.util.BlockRenderLayer;
import net.minecraft.util.EnumBlockRenderType;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumHand;
import net.minecraft.util.IStringSerializable;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.text.translation.I18n;
import net.minecraft.world.ChunkCache;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;
import net.minecraft.world.chunk.Chunk;

public class BlockFlowerPot extends BlockContainer {
	public static final PropertyInteger LEGACY_DATA = PropertyInteger.create("legacy_data", 0, 15);
	public static final PropertyEnum<BlockFlowerPot.EnumFlowerType> CONTENTS = PropertyEnum.<BlockFlowerPot.EnumFlowerType>create(
			"contents", BlockFlowerPot.EnumFlowerType.class);
	protected static final AxisAlignedBB FLOWER_POT_AABB = new AxisAlignedBB(0.3125D, 0.0D, 0.3125D, 0.6875D, 0.375D,
			0.6875D);

	public BlockFlowerPot() {
		super(Material.CIRCUITS);
		this.setDefaultState(this.blockState.getBaseState().withProperty(CONTENTS, BlockFlowerPot.EnumFlowerType.EMPTY)
				.withProperty(LEGACY_DATA, Integer.valueOf(0)));
	}

	/**
	 * Gets the localized name of this block. Used for the statistics page.
	 */
	public String getLocalizedName() {
		return I18n.translateToLocal("item.flowerPot.name");
	}

	public AxisAlignedBB getBoundingBox(IBlockState state, IBlockAccess source, BlockPos pos) {
		return FLOWER_POT_AABB;
	}

	/**
	 * Used to determine ambient occlusion and culling when rebuilding chunks for
	 * render
	 */
	public boolean isOpaqueCube(IBlockState state) {
		return false;
	}

	/**
	 * The type of render function called. MODEL for mixed tesr and static model,
	 * MODELBLOCK_ANIMATED for TESR-only, LIQUID for vanilla liquids, INVISIBLE to
	 * skip all rendering
	 */
	public EnumBlockRenderType getRenderType(IBlockState state) {
		return EnumBlockRenderType.MODEL;
	}

	public boolean isFullCube(IBlockState state) {
		return false;
	}

	public boolean onBlockActivated(World worldIn, BlockPos pos, IBlockState state, EntityPlayer playerIn,
			EnumHand hand, EnumFacing heldItem, float side, float hitX, float hitY) {
		ItemStack itemstack = playerIn.getHeldItem(hand);
		TileEntityFlowerPot tileentityflowerpot = this.getTileEntity(worldIn, pos);

		if (tileentityflowerpot == null) {
			return false;
		} else {
			ItemStack itemstack1 = tileentityflowerpot.getFlowerItemStack();

			if (itemstack1.func_190926_b()) {
				if (!this.func_190951_a(itemstack)) {
					return false;
				}

				tileentityflowerpot.func_190614_a(itemstack);
				playerIn.addStat(StatList.FLOWER_POTTED);

				if (!playerIn.capabilities.isCreativeMode) {
					itemstack.func_190918_g(1);
				}
			} else {
				if (itemstack.func_190926_b()) {
					playerIn.setHeldItem(hand, itemstack1);
				} else if (!playerIn.func_191521_c(itemstack1)) {
					playerIn.dropItem(itemstack1, false);
				}

				tileentityflowerpot.func_190614_a(ItemStack.field_190927_a);
			}

			tileentityflowerpot.markDirty();
			worldIn.notifyBlockUpdate(pos, state, state, 3);
			return true;
		}
	}

	private boolean func_190951_a(ItemStack p_190951_1_) {
		Block block = Block.getBlockFromItem(p_190951_1_.getItem());

		if (block != Blocks.YELLOW_FLOWER && block != Blocks.RED_FLOWER && block != Blocks.CACTUS
				&& block != Blocks.BROWN_MUSHROOM && block != Blocks.RED_MUSHROOM && block != Blocks.SAPLING
				&& block != Blocks.DEADBUSH) {
			int i = p_190951_1_.getMetadata();
			return block == Blocks.TALLGRASS && i == BlockTallGrass.EnumType.FERN.getMeta();
		} else {
			return true;
		}
	}

	public ItemStack getItem(World worldIn, BlockPos pos, IBlockState state) {
		TileEntityFlowerPot tileentityflowerpot = this.getTileEntity(worldIn, pos);

		if (tileentityflowerpot != null) {
			ItemStack itemstack = tileentityflowerpot.getFlowerItemStack();

			if (!itemstack.func_190926_b()) {
				return itemstack;
			}
		}

		return new ItemStack(Items.FLOWER_POT);
	}

	public boolean canPlaceBlockAt(World worldIn, BlockPos pos) {
		return super.canPlaceBlockAt(worldIn, pos) && worldIn.getBlockState(pos.down()).isFullyOpaque();
	}

	/**
	 * Called when a neighboring block was changed and marks that this state should
	 * perform any checks during a neighbor change. Cases may include when redstone
	 * power is updated, cactus blocks popping off due to a neighboring solid block,
	 * etc.
	 */
	public void neighborChanged(IBlockState state, World worldIn, BlockPos pos, Block blockIn, BlockPos p_189540_5_) {
		if (!worldIn.getBlockState(pos.down()).isFullyOpaque()) {
			this.dropBlockAsItem(worldIn, pos, state, 0);
			worldIn.setBlockToAir(pos);
		}
	}

	/**
	 * Called serverside after this block is replaced with another in Chunk, but
	 * before the Tile Entity is updated
	 */
	public void breakBlock(World worldIn, BlockPos pos, IBlockState state) {
		TileEntityFlowerPot tileentityflowerpot = this.getTileEntity(worldIn, pos);

		if (tileentityflowerpot != null && tileentityflowerpot.getFlowerPotItem() != null) {
			spawnAsEntity(worldIn, pos,
					new ItemStack(tileentityflowerpot.getFlowerPotItem(), 1, tileentityflowerpot.getFlowerPotData()));
		}

		super.breakBlock(worldIn, pos, state);
	}

	public void onBlockHarvested(World worldIn, BlockPos pos, IBlockState state, EntityPlayer player) {
		super.onBlockHarvested(worldIn, pos, state, player);

		if (player.capabilities.isCreativeMode) {
			TileEntityFlowerPot tileentityflowerpot = this.getTileEntity(worldIn, pos);

			if (tileentityflowerpot != null) {
				tileentityflowerpot.func_190614_a(ItemStack.field_190927_a);
			}
		}
	}

	/**
	 * Get the Item that this Block should drop when harvested.
	 */
	public Item getItemDropped(IBlockState state, EaglercraftRandom rand, int fortune) {
		return Items.FLOWER_POT;
	}

	@Nullable
	private TileEntityFlowerPot getTileEntity(World worldIn, BlockPos pos) {
		TileEntity tileentity = worldIn.getTileEntity(pos);
		return tileentity instanceof TileEntityFlowerPot ? (TileEntityFlowerPot) tileentity : null;
	}

	/**
	 * Returns a new instance of a block's tile entity class. Called on placing the
	 * block.
	 */
	public TileEntity createNewTileEntity(World worldIn, int meta) {
		Block block = null;
		int i = 0;

		switch (meta) {
		case 1:
			block = Blocks.RED_FLOWER;
			i = BlockFlower.EnumFlowerType.POPPY.getMeta();
			break;

		case 2:
			block = Blocks.YELLOW_FLOWER;
			break;

		case 3:
			block = Blocks.SAPLING;
			i = BlockPlanks.EnumType.OAK.getMetadata();
			break;

		case 4:
			block = Blocks.SAPLING;
			i = BlockPlanks.EnumType.SPRUCE.getMetadata();
			break;

		case 5:
			block = Blocks.SAPLING;
			i = BlockPlanks.EnumType.BIRCH.getMetadata();
			break;

		case 6:
			block = Blocks.SAPLING;
			i = BlockPlanks.EnumType.JUNGLE.getMetadata();
			break;

		case 7:
			block = Blocks.RED_MUSHROOM;
			break;

		case 8:
			block = Blocks.BROWN_MUSHROOM;
			break;

		case 9:
			block = Blocks.CACTUS;
			break;

		case 10:
			block = Blocks.DEADBUSH;
			break;

		case 11:
			block = Blocks.TALLGRASS;
			i = BlockTallGrass.EnumType.FERN.getMeta();
			break;

		case 12:
			block = Blocks.SAPLING;
			i = BlockPlanks.EnumType.ACACIA.getMetadata();
			break;

		case 13:
			block = Blocks.SAPLING;
			i = BlockPlanks.EnumType.DARK_OAK.getMetadata();
		}

		return new TileEntityFlowerPot(Item.getItemFromBlock(block), i);
	}

	protected BlockStateContainer createBlockState() {
		return new BlockStateContainer(this, new IProperty[] { CONTENTS, LEGACY_DATA });
	}

	/**
	 * Convert the BlockState into the correct metadata value
	 */
	public int getMetaFromState(IBlockState state) {
		return ((Integer) state.getValue(LEGACY_DATA)).intValue();
	}

	/**
	 * Get the actual Block state of this Block at the given position. This applies
	 * properties not visible in the metadata, such as fence connections.
	 */
	public IBlockState getActualState(IBlockState state, IBlockAccess worldIn, BlockPos pos) {
		BlockFlowerPot.EnumFlowerType blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.EMPTY;
		TileEntity tileentity = worldIn instanceof ChunkCache
				? ((ChunkCache) worldIn).getTileEntity(pos, Chunk.EnumCreateEntityType.CHECK)
				: worldIn.getTileEntity(pos);

		if (tileentity instanceof TileEntityFlowerPot) {
			TileEntityFlowerPot tileentityflowerpot = (TileEntityFlowerPot) tileentity;
			Item item = tileentityflowerpot.getFlowerPotItem();

			if (item instanceof ItemBlock) {
				int i = tileentityflowerpot.getFlowerPotData();
				Block block = Block.getBlockFromItem(item);

				if (block == Blocks.SAPLING) {
					switch (BlockPlanks.EnumType.byMetadata(i)) {
					case OAK:
						blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.OAK_SAPLING;
						break;

					case SPRUCE:
						blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.SPRUCE_SAPLING;
						break;

					case BIRCH:
						blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.BIRCH_SAPLING;
						break;

					case JUNGLE:
						blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.JUNGLE_SAPLING;
						break;

					case ACACIA:
						blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.ACACIA_SAPLING;
						break;

					case DARK_OAK:
						blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.DARK_OAK_SAPLING;
						break;

					default:
						blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.EMPTY;
					}
				} else if (block == Blocks.TALLGRASS) {
					switch (i) {
					case 0:
						blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.DEAD_BUSH;
						break;

					case 2:
						blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.FERN;
						break;

					default:
						blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.EMPTY;
					}
				} else if (block == Blocks.YELLOW_FLOWER) {
					blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.DANDELION;
				} else if (block == Blocks.RED_FLOWER) {
					switch (BlockFlower.EnumFlowerType.getType(BlockFlower.EnumFlowerColor.RED, i)) {
					case POPPY:
						blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.POPPY;
						break;

					case BLUE_ORCHID:
						blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.BLUE_ORCHID;
						break;

					case ALLIUM:
						blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.ALLIUM;
						break;

					case HOUSTONIA:
						blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.HOUSTONIA;
						break;

					case RED_TULIP:
						blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.RED_TULIP;
						break;

					case ORANGE_TULIP:
						blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.ORANGE_TULIP;
						break;

					case WHITE_TULIP:
						blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.WHITE_TULIP;
						break;

					case PINK_TULIP:
						blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.PINK_TULIP;
						break;

					case OXEYE_DAISY:
						blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.OXEYE_DAISY;
						break;

					default:
						blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.EMPTY;
					}
				} else if (block == Blocks.RED_MUSHROOM) {
					blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.MUSHROOM_RED;
				} else if (block == Blocks.BROWN_MUSHROOM) {
					blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.MUSHROOM_BROWN;
				} else if (block == Blocks.DEADBUSH) {
					blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.DEAD_BUSH;
				} else if (block == Blocks.CACTUS) {
					blockflowerpot$enumflowertype = BlockFlowerPot.EnumFlowerType.CACTUS;
				}
			}
		}

		return state.withProperty(CONTENTS, blockflowerpot$enumflowertype);
	}

	public BlockRenderLayer getBlockLayer() {
		return BlockRenderLayer.CUTOUT;
	}

	public BlockFaceShape func_193383_a(IBlockAccess p_193383_1_, IBlockState p_193383_2_, BlockPos p_193383_3_,
			EnumFacing p_193383_4_) {
		return BlockFaceShape.UNDEFINED;
	}

	public static enum EnumFlowerType implements IStringSerializable {
		EMPTY("empty"), POPPY("rose"), BLUE_ORCHID("blue_orchid"), ALLIUM("allium"), HOUSTONIA("houstonia"),
		RED_TULIP("red_tulip"), ORANGE_TULIP("orange_tulip"), WHITE_TULIP("white_tulip"), PINK_TULIP("pink_tulip"),
		OXEYE_DAISY("oxeye_daisy"), DANDELION("dandelion"), OAK_SAPLING("oak_sapling"),
		SPRUCE_SAPLING("spruce_sapling"), BIRCH_SAPLING("birch_sapling"), JUNGLE_SAPLING("jungle_sapling"),
		ACACIA_SAPLING("acacia_sapling"), DARK_OAK_SAPLING("dark_oak_sapling"), MUSHROOM_RED("mushroom_red"),
		MUSHROOM_BROWN("mushroom_brown"), DEAD_BUSH("dead_bush"), FERN("fern"), CACTUS("cactus");

		private final String name;

		private EnumFlowerType(String name) {
			this.name = name;
		}

		public String toString() {
			return this.name;
		}

		public String getName() {
			return this.name;
		}
	}
}
