package net.minecraft.block;

import net.lax1dude.eaglercraft.EaglercraftRandom;
import net.minecraft.block.material.MapColor;
import net.minecraft.block.material.Material;
import net.minecraft.block.properties.IProperty;
import net.minecraft.block.properties.PropertyEnum;
import net.minecraft.block.state.BlockFaceShape;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.init.Blocks;
import net.minecraft.item.ItemStack;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.IStringSerializable;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;

public abstract class BlockSlab extends Block {
	public static final PropertyEnum<BlockSlab.EnumBlockHalf> HALF = PropertyEnum.<BlockSlab.EnumBlockHalf>create(
			"half", BlockSlab.EnumBlockHalf.class);
	protected static final AxisAlignedBB AABB_BOTTOM_HALF = new AxisAlignedBB(0.0D, 0.0D, 0.0D, 1.0D, 0.5D, 1.0D);
	protected static final AxisAlignedBB AABB_TOP_HALF = new AxisAlignedBB(0.0D, 0.5D, 0.0D, 1.0D, 1.0D, 1.0D);

	public BlockSlab(Material materialIn) {
		this(materialIn, materialIn.getMaterialMapColor());
	}

	public BlockSlab(Material p_i47249_1_, MapColor p_i47249_2_) {
		super(p_i47249_1_, p_i47249_2_);
		this.fullBlock = this.isDouble();
		this.setLightOpacity(255);
	}

	protected boolean canSilkHarvest() {
		return false;
	}

	public AxisAlignedBB getBoundingBox(IBlockState state, IBlockAccess source, BlockPos pos) {
		if (this.isDouble()) {
			return FULL_BLOCK_AABB;
		} else {
			return state.getValue(HALF) == BlockSlab.EnumBlockHalf.TOP ? AABB_TOP_HALF : AABB_BOTTOM_HALF;
		}
	}

	/**
	 * Checks if an IBlockState represents a block that is opaque and a full cube.
	 */
	public boolean isFullyOpaque(IBlockState state) {
		return ((BlockSlab) state.getBlock()).isDouble() || state.getValue(HALF) == BlockSlab.EnumBlockHalf.TOP;
	}

	public BlockFaceShape func_193383_a(IBlockAccess p_193383_1_, IBlockState p_193383_2_, BlockPos p_193383_3_,
			EnumFacing p_193383_4_) {
		if (((BlockSlab) p_193383_2_.getBlock()).isDouble()) {
			return BlockFaceShape.SOLID;
		} else if (p_193383_4_ == EnumFacing.UP && p_193383_2_.getValue(HALF) == BlockSlab.EnumBlockHalf.TOP) {
			return BlockFaceShape.SOLID;
		} else {
			return p_193383_4_ == EnumFacing.DOWN && p_193383_2_.getValue(HALF) == BlockSlab.EnumBlockHalf.BOTTOM
					? BlockFaceShape.SOLID
					: BlockFaceShape.UNDEFINED;
		}
	}

	/**
	 * Used to determine ambient occlusion and culling when rebuilding chunks for
	 * render
	 */
	public boolean isOpaqueCube(IBlockState state) {
		return this.isDouble();
	}

	/**
	 * Called by ItemBlocks just before a block is actually set in the world, to
	 * allow for adjustments to the IBlockstate
	 */
	public IBlockState onBlockPlaced(World worldIn, BlockPos pos, EnumFacing facing, float hitX, float hitY, float hitZ,
			int meta, EntityLivingBase placer) {
		IBlockState iblockstate = super.onBlockPlaced(worldIn, pos, facing, hitX, hitY, hitZ, meta, placer)
				.withProperty(HALF, BlockSlab.EnumBlockHalf.BOTTOM);

		if (this.isDouble()) {
			return iblockstate;
		} else {
			return facing != EnumFacing.DOWN && (facing == EnumFacing.UP || (double) hitY <= 0.5D) ? iblockstate
					: iblockstate.withProperty(HALF, BlockSlab.EnumBlockHalf.TOP);
		}
	}

	/**
	 * Returns the quantity of items to drop on block destruction.
	 */
	public int quantityDropped(EaglercraftRandom random) {
		return this.isDouble() ? 2 : 1;
	}

	public boolean isFullCube(IBlockState state) {
		return this.isDouble();
	}

	public boolean shouldSideBeRendered(IBlockState blockState, IBlockAccess blockAccess, BlockPos pos,
			EnumFacing side) {
		if (this.isDouble()) {
			return super.shouldSideBeRendered(blockState, blockAccess, pos, side);
		} else if (side != EnumFacing.UP && side != EnumFacing.DOWN
				&& !super.shouldSideBeRendered(blockState, blockAccess, pos, side)) {
			return false;
		} else {
			IBlockState iblockstate = blockAccess.getBlockState(pos.offset(side));
			boolean flag = isHalfSlab(iblockstate) && iblockstate.getValue(HALF) == BlockSlab.EnumBlockHalf.TOP;
			boolean flag1 = isHalfSlab(blockState) && blockState.getValue(HALF) == BlockSlab.EnumBlockHalf.TOP;

			if (flag1) {
				if (side == EnumFacing.DOWN) {
					return true;
				} else if (side == EnumFacing.UP && super.shouldSideBeRendered(blockState, blockAccess, pos, side)) {
					return true;
				} else {
					return !isHalfSlab(iblockstate) || !flag;
				}
			} else if (side == EnumFacing.UP) {
				return true;
			} else if (side == EnumFacing.DOWN && super.shouldSideBeRendered(blockState, blockAccess, pos, side)) {
				return true;
			} else {
				return !isHalfSlab(iblockstate) || flag;
			}
		}
	}

	protected static boolean isHalfSlab(IBlockState state) {
		Block block = state.getBlock();
		return block == Blocks.STONE_SLAB || block == Blocks.WOODEN_SLAB || block == Blocks.STONE_SLAB2
				|| block == Blocks.PURPUR_SLAB;
	}

	/**
	 * Returns the slab block name with the type associated with it
	 */
	public abstract String getUnlocalizedName(int meta);

	public abstract boolean isDouble();

	public abstract IProperty<?> getVariantProperty();

	public abstract Comparable<?> getTypeForItem(ItemStack stack);

	public static enum EnumBlockHalf implements IStringSerializable {
		TOP("top"), BOTTOM("bottom");

		private final String name;

		private EnumBlockHalf(String name) {
			this.name = name;
		}

		public String toString() {
			return this.name;
		}

		public String getName() {
			return this.name;
		}
	}
	
	public boolean alfheim$useNeighborBrightness(final IBlockState blockState, final EnumFacing facing,
			final IBlockAccess blockAccess, final BlockPos blockPos) {
		if (blockState.isFullCube())
			return false;

		if (facing.getAxis() != EnumFacing.Axis.Y)
			return true;

		return facing == (blockState.getValue(HALF) == EnumBlockHalf.TOP ? EnumFacing.DOWN : EnumFacing.UP);
	}
}
