package net.minecraft.block;

import net.lax1dude.eaglercraft.EaglercraftRandom;
import javax.annotation.Nullable;
import net.minecraft.block.material.Material;
import net.minecraft.block.properties.IProperty;
import net.minecraft.block.properties.PropertyDirection;
import net.minecraft.block.properties.PropertyInteger;
import net.minecraft.block.state.BlockStateContainer;
import net.minecraft.block.state.IBlockState;
import net.minecraft.creativetab.CreativeTabs;
import net.minecraft.init.Blocks;
import net.minecraft.init.Items;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.MathHelper;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;

public class BlockStem extends BlockBush implements IGrowable {
	public static final PropertyInteger AGE = PropertyInteger.create("age", 0, 7);
	public static final PropertyDirection FACING = BlockTorch.FACING;
	private final Block crop;
	protected static final AxisAlignedBB[] STEM_AABB = new AxisAlignedBB[] {
			new AxisAlignedBB(0.375D, 0.0D, 0.375D, 0.625D, 0.125D, 0.625D),
			new AxisAlignedBB(0.375D, 0.0D, 0.375D, 0.625D, 0.25D, 0.625D),
			new AxisAlignedBB(0.375D, 0.0D, 0.375D, 0.625D, 0.375D, 0.625D),
			new AxisAlignedBB(0.375D, 0.0D, 0.375D, 0.625D, 0.5D, 0.625D),
			new AxisAlignedBB(0.375D, 0.0D, 0.375D, 0.625D, 0.625D, 0.625D),
			new AxisAlignedBB(0.375D, 0.0D, 0.375D, 0.625D, 0.75D, 0.625D),
			new AxisAlignedBB(0.375D, 0.0D, 0.375D, 0.625D, 0.875D, 0.625D),
			new AxisAlignedBB(0.375D, 0.0D, 0.375D, 0.625D, 1.0D, 0.625D) };

	protected BlockStem(Block crop) {
		this.setDefaultState(this.blockState.getBaseState().withProperty(AGE, Integer.valueOf(0)).withProperty(FACING,
				EnumFacing.UP));
		this.crop = crop;
		this.setTickRandomly(true);
		this.setCreativeTab((CreativeTabs) null);
	}

	public AxisAlignedBB getBoundingBox(IBlockState state, IBlockAccess source, BlockPos pos) {
		return STEM_AABB[((Integer) state.getValue(AGE)).intValue()];
	}

	/**
	 * Get the actual Block state of this Block at the given position. This applies
	 * properties not visible in the metadata, such as fence connections.
	 */
	public IBlockState getActualState(IBlockState state, IBlockAccess worldIn, BlockPos pos) {
		int i = ((Integer) state.getValue(AGE)).intValue();
		state = state.withProperty(FACING, EnumFacing.UP);

		for (EnumFacing enumfacing : EnumFacing.Plane.HORIZONTAL) {
			if (worldIn.getBlockState(pos.offset(enumfacing)).getBlock() == this.crop && i == 7) {
				state = state.withProperty(FACING, enumfacing);
				break;
			}
		}

		return state;
	}

	/**
	 * Return true if the block can sustain a Bush
	 */
	protected boolean canSustainBush(IBlockState state) {
		return state.getBlock() == Blocks.FARMLAND;
	}

	public void updateTick(World worldIn, BlockPos pos, IBlockState state, EaglercraftRandom rand) {
		super.updateTick(worldIn, pos, state, rand);

		if (worldIn.getLightFromNeighbors(pos.up()) >= 9) {
			float f = BlockCrops.getGrowthChance(this, worldIn, pos);

			if (rand.nextInt((int) (25.0F / f) + 1) == 0) {
				int i = ((Integer) state.getValue(AGE)).intValue();

				if (i < 7) {
					state = state.withProperty(AGE, Integer.valueOf(i + 1));
					worldIn.setBlockState(pos, state, 2);
				} else {
					for (EnumFacing enumfacing : EnumFacing.Plane.HORIZONTAL) {
						if (worldIn.getBlockState(pos.offset(enumfacing)).getBlock() == this.crop) {
							return;
						}
					}

					pos = pos.offset(EnumFacing.Plane.HORIZONTAL.random(rand));
					Block block = worldIn.getBlockState(pos.down()).getBlock();

					if (worldIn.getBlockState(pos).getBlock().blockMaterial == Material.AIR
							&& (block == Blocks.FARMLAND || block == Blocks.DIRT || block == Blocks.GRASS)) {
						worldIn.setBlockState(pos, this.crop.getDefaultState());
					}
				}
			}
		}
	}

	public void growStem(World worldIn, BlockPos pos, IBlockState state) {
		int i = ((Integer) state.getValue(AGE)).intValue() + MathHelper.getInt(worldIn.rand, 2, 5);
		worldIn.setBlockState(pos, state.withProperty(AGE, Integer.valueOf(Math.min(7, i))), 2);
	}

	/**
	 * Spawns this Block's drops into the World as EntityItems.
	 */
	public void dropBlockAsItemWithChance(World worldIn, BlockPos pos, IBlockState state, float chance, int fortune) {
		super.dropBlockAsItemWithChance(worldIn, pos, state, chance, fortune);

		if (!worldIn.isRemote) {
			Item item = this.getSeedItem();

			if (item != null) {
				int i = ((Integer) state.getValue(AGE)).intValue();

				for (int j = 0; j < 3; ++j) {
					if (worldIn.rand.nextInt(15) <= i) {
						spawnAsEntity(worldIn, pos, new ItemStack(item));
					}
				}
			}
		}
	}

	@Nullable
	protected Item getSeedItem() {
		if (this.crop == Blocks.PUMPKIN) {
			return Items.PUMPKIN_SEEDS;
		} else {
			return this.crop == Blocks.MELON_BLOCK ? Items.MELON_SEEDS : null;
		}
	}

	/**
	 * Get the Item that this Block should drop when harvested.
	 */
	public Item getItemDropped(IBlockState state, EaglercraftRandom rand, int fortune) {
		return Items.AIR;
	}

	public ItemStack getItem(World worldIn, BlockPos pos, IBlockState state) {
		Item item = this.getSeedItem();
		return item == null ? ItemStack.field_190927_a : new ItemStack(item);
	}

	/**
	 * Whether this IGrowable can grow
	 */
	public boolean canGrow(World worldIn, BlockPos pos, IBlockState state, boolean isClient) {
		return ((Integer) state.getValue(AGE)).intValue() != 7;
	}

	public boolean canUseBonemeal(World worldIn, EaglercraftRandom rand, BlockPos pos, IBlockState state) {
		return true;
	}

	public void grow(World worldIn, EaglercraftRandom rand, BlockPos pos, IBlockState state) {
		this.growStem(worldIn, pos, state);
	}

	/**
	 * Convert the given metadata into a BlockState for this Block
	 */
	public IBlockState getStateFromMeta(int meta) {
		return this.getDefaultState().withProperty(AGE, Integer.valueOf(meta));
	}

	/**
	 * Convert the BlockState into the correct metadata value
	 */
	public int getMetaFromState(IBlockState state) {
		return ((Integer) state.getValue(AGE)).intValue();
	}

	protected BlockStateContainer createBlockState() {
		return new BlockStateContainer(this, new IProperty[] { AGE, FACING });
	}
}
