package net.minecraft.client.entity;

import com.mojang.authlib.GameProfile;
import net.minecraft.client.Minecraft;
import net.minecraft.util.DamageSource;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.MathHelper;
import net.minecraft.util.text.ITextComponent;
import net.minecraft.world.World;

public class EntityOtherPlayerMP extends AbstractClientPlayer {
	private int otherPlayerMPPosRotationIncrements;
	private double otherPlayerMPX;
	private double otherPlayerMPY;
	private double otherPlayerMPZ;
	private double otherPlayerMPYaw;
	private double otherPlayerMPPitch;

	public EntityOtherPlayerMP(World worldIn, GameProfile gameProfileIn) {
		super(worldIn, gameProfileIn);
		this.stepHeight = 1.0F;
		this.noClip = true;
		this.renderOffsetY = 0.25F;
	}

	/**
	 * Checks if the entity is in range to render.
	 */
	public boolean isInRangeToRenderDist(double distance) {
		double d0 = this.getEntityBoundingBox().getAverageEdgeLength() * 10.0D;

		if (Double.isNaN(d0)) {
			d0 = 1.0D;
		}

		d0 = d0 * 64.0D * getRenderDistanceWeight();
		return distance < d0 * d0;
	}

	/**
	 * Called when the entity is attacked.
	 */
	public boolean attackEntityFrom(DamageSource source, float amount) {
		return true;
	}

	/**
	 * Set the position and rotation values directly without any clamping.
	 */
	public void setPositionAndRotationDirect(double x, double y, double z, float yaw, float pitch,
			int posRotationIncrements, boolean teleport) {
		this.otherPlayerMPX = x;
		this.otherPlayerMPY = y;
		this.otherPlayerMPZ = z;
		this.otherPlayerMPYaw = (double) yaw;
		this.otherPlayerMPPitch = (double) pitch;
		this.otherPlayerMPPosRotationIncrements = posRotationIncrements;
	}

	/**
	 * Called to update the entity's position/logic.
	 */
	public void onUpdate() {
		this.renderOffsetY = 0.0F;
		super.onUpdate();
		this.prevLimbSwingAmount = this.limbSwingAmount;
		double d0 = this.posX - this.prevPosX;
		double d1 = this.posZ - this.prevPosZ;
		float f = MathHelper.sqrt(d0 * d0 + d1 * d1) * 4.0F;

		if (f > 1.0F) {
			f = 1.0F;
		}

		this.limbSwingAmount += (f - this.limbSwingAmount) * 0.4F;
		this.limbSwing += this.limbSwingAmount;
	}

	/**
	 * Called frequently so the entity can update its state every tick as required.
	 * For example, zombies and skeletons use this to react to sunlight and start to
	 * burn.
	 */
	public void onLivingUpdate() {
		if (this.otherPlayerMPPosRotationIncrements > 0) {
			double d0 = this.posX
					+ (this.otherPlayerMPX - this.posX) / (double) this.otherPlayerMPPosRotationIncrements;
			double d1 = this.posY
					+ (this.otherPlayerMPY - this.posY) / (double) this.otherPlayerMPPosRotationIncrements;
			double d2 = this.posZ
					+ (this.otherPlayerMPZ - this.posZ) / (double) this.otherPlayerMPPosRotationIncrements;
			double d3;

			for (d3 = this.otherPlayerMPYaw - (double) this.rotationYaw; d3 < -180.0D; d3 += 360.0D) {
				;
			}

			while (d3 >= 180.0D) {
				d3 -= 360.0D;
			}

			this.rotationYaw = (float) ((double) this.rotationYaw
					+ d3 / (double) this.otherPlayerMPPosRotationIncrements);
			this.rotationPitch = (float) ((double) this.rotationPitch
					+ (this.otherPlayerMPPitch - (double) this.rotationPitch)
							/ (double) this.otherPlayerMPPosRotationIncrements);
			--this.otherPlayerMPPosRotationIncrements;
			this.setPosition(d0, d1, d2);
			this.setRotation(this.rotationYaw, this.rotationPitch);
		}

		this.prevCameraYaw = this.cameraYaw;
		this.updateArmSwingProgress();
		float f1 = MathHelper.sqrt(this.motionX * this.motionX + this.motionZ * this.motionZ);
		float f = (float) Math.atan(-this.motionY * 0.20000000298023224D) * 15.0F;

		if (f1 > 0.1F) {
			f1 = 0.1F;
		}

		if (!this.onGround || this.getHealth() <= 0.0F) {
			f1 = 0.0F;
		}

		if (this.onGround || this.getHealth() <= 0.0F) {
			f = 0.0F;
		}

		this.cameraYaw += (f1 - this.cameraYaw) * 0.4F;
		this.cameraPitch += (f - this.cameraPitch) * 0.8F;
		this.collideWithNearbyEntities();
	}

	/**
	 * Send a chat message to the CommandSender
	 */
	public void addChatMessage(ITextComponent component) {
		Minecraft.getMinecraft().ingameGUI.getChatGUI().printChatMessage(component);
	}

	/**
	 * Returns {@code true} if the CommandSender is allowed to execute the command,
	 * {@code false} if not
	 */
	public boolean canCommandSenderUseCommand(int permLevel, String commandName) {
		return false;
	}

	/**
	 * Get the position in the world. <b>{@code null} is not allowed!</b> If you are
	 * not an entity in the world, return the coordinates 0, 0, 0
	 */
	public BlockPos getPosition() {
		return new BlockPos(this.posX + 0.5D, this.posY + 0.5D, this.posZ + 0.5D);
	}
}
