package net.minecraft.client.model;

import com.google.common.collect.Lists;
import java.util.List;

import net.lax1dude.eaglercraft.opengl.WorldRenderer;
import net.lax1dude.eaglercraft.opengl.EaglercraftGPU;
import net.lax1dude.eaglercraft.opengl.GlStateManager;
import net.minecraft.client.renderer.GLAllocation;
import net.minecraft.client.renderer.Tessellator;

public class ModelRenderer {
	/** The size of the texture file's width in pixels. */
	public float textureWidth;

	/** The size of the texture file's height in pixels. */
	public float textureHeight;

	/** The X offset into the texture used for displaying this model */
	private int textureOffsetX;

	/** The Y offset into the texture used for displaying this model */
	private int textureOffsetY;
	public float rotationPointX;
	public float rotationPointY;
	public float rotationPointZ;
	public float rotateAngleX;
	public float rotateAngleY;
	public float rotateAngleZ;
	private boolean compiled;

	/** The GL display list rendered by the Tessellator for this model */
	private int displayList;
	public boolean mirror;
	public boolean showModel;

	/** Hides the model. */
	public boolean isHidden;
	public List<ModelBox> cubeList;
	public List<ModelRenderer> childModels;
	public final String boxName;
	private final ModelBase baseModel;
	public float offsetX;
	public float offsetY;
	public float offsetZ;

	public ModelRenderer(ModelBase model, String boxNameIn) {
		this.textureWidth = 64.0F;
		this.textureHeight = 32.0F;
		this.showModel = true;
		this.cubeList = Lists.<ModelBox>newArrayList();
		this.baseModel = model;
		model.boxList.add(this);
		this.boxName = boxNameIn;
		this.setTextureSize(model.textureWidth, model.textureHeight);
	}

	public ModelRenderer(ModelBase model) {
		this(model, (String) null);
	}

	public ModelRenderer(ModelBase model, int texOffX, int texOffY) {
		this(model);
		this.setTextureOffset(texOffX, texOffY);
	}

	/**
	 * Sets the current box's rotation points and rotation angles to another box.
	 */
	public void addChild(ModelRenderer renderer) {
		if (this.childModels == null) {
			this.childModels = Lists.<ModelRenderer>newArrayList();
		}

		this.childModels.add(renderer);
	}

	public ModelRenderer setTextureOffset(int x, int y) {
		this.textureOffsetX = x;
		this.textureOffsetY = y;
		return this;
	}

	public ModelRenderer addBox(String partName, float offX, float offY, float offZ, int width, int height, int depth) {
		partName = this.boxName + "." + partName;
		TextureOffset textureoffset = this.baseModel.getTextureOffset(partName);
		this.setTextureOffset(textureoffset.textureOffsetX, textureoffset.textureOffsetY);
		this.cubeList.add((new ModelBox(this, this.textureOffsetX, this.textureOffsetY, offX, offY, offZ, width, height,
				depth, 0.0F)).setBoxName(partName));
		return this;
	}

	public ModelRenderer addBox(float offX, float offY, float offZ, int width, int height, int depth) {
		this.cubeList.add(new ModelBox(this, this.textureOffsetX, this.textureOffsetY, offX, offY, offZ, width, height,
				depth, 0.0F));
		return this;
	}

	public ModelRenderer addBox(float offX, float offY, float offZ, int width, int height, int depth,
			boolean mirrored) {
		this.cubeList.add(new ModelBox(this, this.textureOffsetX, this.textureOffsetY, offX, offY, offZ, width, height,
				depth, 0.0F, mirrored));
		return this;
	}

	/**
	 * Creates a textured box.
	 */
	public void addBox(float offX, float offY, float offZ, int width, int height, int depth, float scaleFactor) {
		this.cubeList.add(new ModelBox(this, this.textureOffsetX, this.textureOffsetY, offX, offY, offZ, width, height,
				depth, scaleFactor));
	}

	public void setRotationPoint(float rotationPointXIn, float rotationPointYIn, float rotationPointZIn) {
		this.rotationPointX = rotationPointXIn;
		this.rotationPointY = rotationPointYIn;
		this.rotationPointZ = rotationPointZIn;
	}

	public void render(float scale) {
		if (!this.isHidden) {
			if (this.showModel) {
				if (!this.compiled) {
					this.compileDisplayList(scale);
				}

				GlStateManager.translate(this.offsetX, this.offsetY, this.offsetZ);

				if (this.rotateAngleX == 0.0F && this.rotateAngleY == 0.0F && this.rotateAngleZ == 0.0F) {
					if (this.rotationPointX == 0.0F && this.rotationPointY == 0.0F && this.rotationPointZ == 0.0F) {
						GlStateManager.callList(this.displayList);

						if (this.childModels != null) {
							for (int k = 0; k < this.childModels.size(); ++k) {
								((ModelRenderer) this.childModels.get(k)).render(scale);
							}
						}
					} else {
						GlStateManager.translate(this.rotationPointX * scale, this.rotationPointY * scale,
								this.rotationPointZ * scale);
						GlStateManager.callList(this.displayList);

						if (this.childModels != null) {
							for (int j = 0; j < this.childModels.size(); ++j) {
								((ModelRenderer) this.childModels.get(j)).render(scale);
							}
						}

						GlStateManager.translate(-this.rotationPointX * scale, -this.rotationPointY * scale,
								-this.rotationPointZ * scale);
					}
				} else {
					GlStateManager.pushMatrix();
					GlStateManager.translate(this.rotationPointX * scale, this.rotationPointY * scale,
							this.rotationPointZ * scale);

					GlStateManager.rotateZYXRad(this.rotateAngleX, this.rotateAngleY, this.rotateAngleZ);

					GlStateManager.callList(this.displayList);

					if (this.childModels != null) {
						for (int i = 0; i < this.childModels.size(); ++i) {
							((ModelRenderer) this.childModels.get(i)).render(scale);
						}
					}

					GlStateManager.popMatrix();
				}

				GlStateManager.translate(-this.offsetX, -this.offsetY, -this.offsetZ);
			}
		}
	}

	public void renderWithRotation(float scale) {
		if (!this.isHidden) {
			if (this.showModel) {
				if (!this.compiled) {
					this.compileDisplayList(scale);
				}

				GlStateManager.pushMatrix();
				GlStateManager.translate(this.rotationPointX * scale, this.rotationPointY * scale,
						this.rotationPointZ * scale);

				GlStateManager.rotateZYXRad(this.rotateAngleX, this.rotateAngleY, this.rotateAngleZ);

				GlStateManager.callList(this.displayList);
				GlStateManager.popMatrix();
			}
		}
	}

	/**
	 * Allows the changing of Angles after a box has been rendered
	 */
	public void postRender(float scale) {
		if (!this.isHidden) {
			if (this.showModel) {
				if (!this.compiled) {
					this.compileDisplayList(scale);
				}

				if (this.rotateAngleX == 0.0F && this.rotateAngleY == 0.0F && this.rotateAngleZ == 0.0F) {
					if (this.rotationPointX != 0.0F || this.rotationPointY != 0.0F || this.rotationPointZ != 0.0F) {
						GlStateManager.translate(this.rotationPointX * scale, this.rotationPointY * scale,
								this.rotationPointZ * scale);
					}
				} else {
					GlStateManager.translate(this.rotationPointX * scale, this.rotationPointY * scale,
							this.rotationPointZ * scale);

					GlStateManager.rotateZYXRad(this.rotateAngleX, this.rotateAngleY, this.rotateAngleZ);
				}
			}
		}
	}

	/**
	 * Compiles a GL display list for this model
	 */
	private void compileDisplayList(float scale) {
		this.displayList = GLAllocation.generateDisplayLists();
		EaglercraftGPU.glNewList(this.displayList, 4864);
		WorldRenderer bufferbuilder = Tessellator.getInstance().getBuffer();

		for (int i = 0; i < this.cubeList.size(); ++i) {
			((ModelBox) this.cubeList.get(i)).render(bufferbuilder, scale);
		}

		EaglercraftGPU.glEndList();
		this.compiled = true;
	}

	/**
	 * Returns the model renderer with the new texture parameters.
	 */
	public ModelRenderer setTextureSize(int textureWidthIn, int textureHeightIn) {
		this.textureWidth = (float) textureWidthIn;
		this.textureHeight = (float) textureHeightIn;
		return this;
	}
}
