package net.minecraft.entity.ai;

import net.minecraft.entity.EntityCreature;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.util.math.Vec3d;

public class EntityAIMoveTowardsTarget extends EntityAIBase {
	private final EntityCreature theEntity;
	private EntityLivingBase targetEntity;
	private double movePosX;
	private double movePosY;
	private double movePosZ;
	private final double speed;

	/**
	 * If the distance to the target entity is further than this, this AI task will
	 * not run.
	 */
	private final float maxTargetDistance;

	public EntityAIMoveTowardsTarget(EntityCreature creature, double speedIn, float targetMaxDistance) {
		this.theEntity = creature;
		this.speed = speedIn;
		this.maxTargetDistance = targetMaxDistance;
		this.setMutexBits(1);
	}

	/**
	 * Returns whether the EntityAIBase should begin execution.
	 */
	public boolean shouldExecute() {
		this.targetEntity = this.theEntity.getAttackTarget();

		if (this.targetEntity == null) {
			return false;
		} else if (this.targetEntity
				.getDistanceSqToEntity(this.theEntity) > (double) (this.maxTargetDistance * this.maxTargetDistance)) {
			return false;
		} else {
			Vec3d vec3d = RandomPositionGenerator.findRandomTargetBlockTowards(this.theEntity, 16, 7,
					new Vec3d(this.targetEntity.posX, this.targetEntity.posY, this.targetEntity.posZ));

			if (vec3d == null) {
				return false;
			} else {
				this.movePosX = vec3d.xCoord;
				this.movePosY = vec3d.yCoord;
				this.movePosZ = vec3d.zCoord;
				return true;
			}
		}
	}

	/**
	 * Returns whether an in-progress EntityAIBase should continue executing
	 */
	public boolean continueExecuting() {
		return !this.theEntity.getNavigator().noPath() && this.targetEntity.isEntityAlive() && this.targetEntity
				.getDistanceSqToEntity(this.theEntity) < (double) (this.maxTargetDistance * this.maxTargetDistance);
	}

	/**
	 * Resets the task
	 */
	public void resetTask() {
		this.targetEntity = null;
	}

	/**
	 * Execute a one shot task or start executing a continuous task
	 */
	public void startExecuting() {
		this.theEntity.getNavigator().tryMoveToXYZ(this.movePosX, this.movePosY, this.movePosZ, this.speed);
	}
}
