package net.minecraft.entity.item;

import javax.annotation.Nullable;
import net.minecraft.block.material.Material;
import net.minecraft.entity.Entity;
import net.minecraft.entity.MoverType;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.init.Items;
import net.minecraft.init.SoundEvents;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.datasync.DataParameter;
import net.minecraft.network.datasync.DataSerializers;
import net.minecraft.network.datasync.EntityDataManager;
import net.minecraft.stats.StatList;
import net.minecraft.util.DamageSource;
import net.minecraft.util.datafix.DataFixer;
import net.minecraft.util.datafix.FixTypes;
import net.minecraft.util.datafix.walkers.ItemStackData;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.MathHelper;
import net.minecraft.util.text.translation.I18n;
import net.minecraft.world.World;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class EntityItem extends Entity {
	private static final Logger LOGGER = LogManager.getLogger();
	private static final DataParameter<ItemStack> ITEM = EntityDataManager.<ItemStack>createKey(6,
			DataSerializers.OPTIONAL_ITEM_STACK);

	/**
	 * The age of this EntityItem (used to animate it up and down as well as expire
	 * it)
	 */
	private int age;
	private int delayBeforeCanPickup;

	/** The health of this EntityItem. (For example, damage for tools) */
	private int health;
	private String thrower;
	private String owner;

	/** The EntityItem's random initial float height. */
	public float hoverStart;

	public EntityItem(World worldIn, double x, double y, double z) {
		super(worldIn);
		this.health = 5;
		this.hoverStart = (float) (Math.random() * Math.PI * 2.0D);
		this.setSize(0.25F, 0.25F);
		this.setPosition(x, y, z);
		this.rotationYaw = (float) (Math.random() * 360.0D);
		this.motionX = (double) ((float) (Math.random() * 0.20000000298023224D - 0.10000000149011612D));
		this.motionY = 0.20000000298023224D;
		this.motionZ = (double) ((float) (Math.random() * 0.20000000298023224D - 0.10000000149011612D));
	}

	public EntityItem(World worldIn, double x, double y, double z, ItemStack stack) {
		this(worldIn, x, y, z);
		this.setEntityItemStack(stack);
	}

	/**
	 * returns if this entity triggers Block.onEntityWalking on the blocks they walk
	 * on. used for spiders and wolves to prevent them from trampling crops
	 */
	protected boolean canTriggerWalking() {
		return false;
	}

	public EntityItem(World worldIn) {
		super(worldIn);
		this.health = 5;
		this.hoverStart = (float) (Math.random() * Math.PI * 2.0D);
		this.setSize(0.25F, 0.25F);
		this.setEntityItemStack(ItemStack.field_190927_a);
	}

	protected void entityInit() {
		this.getDataManager().register(ITEM, ItemStack.field_190927_a);
	}

	/**
	 * Called to update the entity's position/logic.
	 */
	public void onUpdate() {
		if (this.getEntityItem().func_190926_b()) {
			this.setDead();
		} else {
			super.onUpdate();

			if (this.delayBeforeCanPickup > 0 && this.delayBeforeCanPickup != 32767) {
				--this.delayBeforeCanPickup;
			}

			this.prevPosX = this.posX;
			this.prevPosY = this.posY;
			this.prevPosZ = this.posZ;
			double d0 = this.motionX;
			double d1 = this.motionY;
			double d2 = this.motionZ;

			if (!this.hasNoGravity()) {
				this.motionY -= 0.03999999910593033D;
			}

			if (this.world.isRemote) {
				this.noClip = false;
			} else {
				this.noClip = this.pushOutOfBlocks(this.posX,
						(this.getEntityBoundingBox().minY + this.getEntityBoundingBox().maxY) / 2.0D, this.posZ);
			}

			this.moveEntity(MoverType.SELF, this.motionX, this.motionY, this.motionZ);
			boolean flag = (int) this.prevPosX != (int) this.posX || (int) this.prevPosY != (int) this.posY
					|| (int) this.prevPosZ != (int) this.posZ;

			if (flag || this.ticksExisted % 25 == 0) {
				if (this.world.getBlockState(new BlockPos(this)).getMaterial() == Material.LAVA) {
					this.motionY = 0.20000000298023224D;
					this.motionX = (double) ((this.rand.nextFloat() - this.rand.nextFloat()) * 0.2F);
					this.motionZ = (double) ((this.rand.nextFloat() - this.rand.nextFloat()) * 0.2F);
					this.playSound(SoundEvents.ENTITY_GENERIC_BURN, 0.4F, 2.0F + this.rand.nextFloat() * 0.4F);
				}

				if (!this.world.isRemote) {
					this.searchForOtherItemsNearby();
				}
			}

			float f = 0.98F;

			if (this.onGround) {
				f = this.world
						.getBlockState(new BlockPos(MathHelper.floor(this.posX),
								MathHelper.floor(this.getEntityBoundingBox().minY) - 1, MathHelper.floor(this.posZ)))
						.getBlock().slipperiness * 0.98F;
			}

			this.motionX *= (double) f;
			this.motionY *= 0.9800000190734863D;
			this.motionZ *= (double) f;

			if (this.onGround) {
				this.motionY *= -0.5D;
			}

			if (this.age != -32768) {
				++this.age;
			}

			this.handleWaterMovement();

			if (!this.world.isRemote) {
				double d3 = this.motionX - d0;
				double d4 = this.motionY - d1;
				double d5 = this.motionZ - d2;
				double d6 = d3 * d3 + d4 * d4 + d5 * d5;

				if (d6 > 0.01D) {
					this.isAirBorne = true;
				}
			}

			if (!this.world.isRemote && this.age >= 6000) {
				this.setDead();
			}
		}
	}

	/**
	 * Looks for other itemstacks nearby and tries to stack them together
	 */
	private void searchForOtherItemsNearby() {
		for (EntityItem entityitem : this.world.getEntitiesWithinAABB(EntityItem.class,
				this.getEntityBoundingBox().expand(0.5D, 0.0D, 0.5D))) {
			this.combineItems(entityitem);
		}
	}

	/**
	 * Tries to merge this item with the item passed as the parameter. Returns true
	 * if successful. Either this item or the other item will be removed from the
	 * world.
	 */
	private boolean combineItems(EntityItem other) {
		if (other == this) {
			return false;
		} else if (other.isEntityAlive() && this.isEntityAlive()) {
			ItemStack itemstack = this.getEntityItem();
			ItemStack itemstack1 = other.getEntityItem();

			if (this.delayBeforeCanPickup != 32767 && other.delayBeforeCanPickup != 32767) {
				if (this.age != -32768 && other.age != -32768) {
					if (itemstack1.getItem() != itemstack.getItem()) {
						return false;
					} else if (itemstack1.hasTagCompound() ^ itemstack.hasTagCompound()) {
						return false;
					} else if (itemstack1.hasTagCompound()
							&& !itemstack1.getTagCompound().equals(itemstack.getTagCompound())) {
						return false;
					} else if (itemstack1.getItem() == null) {
						return false;
					} else if (itemstack1.getItem().getHasSubtypes()
							&& itemstack1.getMetadata() != itemstack.getMetadata()) {
						return false;
					} else if (itemstack1.func_190916_E() < itemstack.func_190916_E()) {
						return other.combineItems(this);
					} else if (itemstack1.func_190916_E() + itemstack.func_190916_E() > itemstack1.getMaxStackSize()) {
						return false;
					} else {
						itemstack1.func_190917_f(itemstack.func_190916_E());
						other.delayBeforeCanPickup = Math.max(other.delayBeforeCanPickup, this.delayBeforeCanPickup);
						other.age = Math.min(other.age, this.age);
						other.setEntityItemStack(itemstack1);
						this.setDead();
						return true;
					}
				} else {
					return false;
				}
			} else {
				return false;
			}
		} else {
			return false;
		}
	}

	/**
	 * sets the age of the item so that it'll despawn one minute after it has been
	 * dropped (instead of five). Used when items are dropped from players in
	 * creative mode
	 */
	public void setAgeToCreativeDespawnTime() {
		this.age = 4800;
	}

	/**
	 * Returns if this entity is in water and will end up adding the waters velocity
	 * to the entity
	 */
	public boolean handleWaterMovement() {
		if (this.world.handleMaterialAcceleration(this.getEntityBoundingBox(), Material.WATER, this)) {
			if (!this.inWater && !this.firstUpdate) {
				this.resetHeight();
			}

			this.inWater = true;
		} else {
			this.inWater = false;
		}

		return this.inWater;
	}

	/**
	 * Will deal the specified amount of fire damage to the entity if the entity
	 * isn't immune to fire damage.
	 */
	protected void dealFireDamage(int amount) {
		this.attackEntityFrom(DamageSource.inFire, (float) amount);
	}

	/**
	 * Called when the entity is attacked.
	 */
	public boolean attackEntityFrom(DamageSource source, float amount) {
		if (this.isEntityInvulnerable(source)) {
			return false;
		} else if (!this.getEntityItem().func_190926_b() && this.getEntityItem().getItem() == Items.NETHER_STAR
				&& source.isExplosion()) {
			return false;
		} else {
			this.setBeenAttacked();
			this.health = (int) ((float) this.health - amount);

			if (this.health <= 0) {
				this.setDead();
			}

			return false;
		}
	}

	public static void registerFixesItem(DataFixer fixer) {
		fixer.registerWalker(FixTypes.ENTITY, new ItemStackData(EntityItem.class, new String[] { "Item" }));
	}

	/**
	 * (abstract) Protected helper method to write subclass entity data to NBT.
	 */
	public void writeEntityToNBT(NBTTagCompound compound) {
		compound.setShort("Health", (short) this.health);
		compound.setShort("Age", (short) this.age);
		compound.setShort("PickupDelay", (short) this.delayBeforeCanPickup);

		if (this.getThrower() != null) {
			compound.setString("Thrower", this.thrower);
		}

		if (this.getOwner() != null) {
			compound.setString("Owner", this.owner);
		}

		if (!this.getEntityItem().func_190926_b()) {
			compound.setTag("Item", this.getEntityItem().writeToNBT(new NBTTagCompound()));
		}
	}

	/**
	 * (abstract) Protected helper method to read subclass entity data from NBT.
	 */
	public void readEntityFromNBT(NBTTagCompound compound) {
		this.health = compound.getShort("Health");
		this.age = compound.getShort("Age");

		if (compound.hasKey("PickupDelay")) {
			this.delayBeforeCanPickup = compound.getShort("PickupDelay");
		}

		if (compound.hasKey("Owner")) {
			this.owner = compound.getString("Owner");
		}

		if (compound.hasKey("Thrower")) {
			this.thrower = compound.getString("Thrower");
		}

		NBTTagCompound nbttagcompound = compound.getCompoundTag("Item");
		this.setEntityItemStack(new ItemStack(nbttagcompound));

		if (this.getEntityItem().func_190926_b()) {
			this.setDead();
		}
	}

	/**
	 * Called by a player entity when they collide with an entity
	 */
	public void onCollideWithPlayer(EntityPlayer entityIn) {
		if (!this.world.isRemote) {
			ItemStack itemstack = this.getEntityItem();
			Item item = itemstack.getItem();
			int i = itemstack.func_190916_E();

			if (this.delayBeforeCanPickup == 0
					&& (this.owner == null || 6000 - this.age <= 200 || this.owner.equals(entityIn.getName()))
					&& entityIn.inventory.addItemStackToInventory(itemstack)) {
				entityIn.onItemPickup(this, i);

				if (itemstack.func_190926_b()) {
					this.setDead();
					itemstack.func_190920_e(i);
				}

				entityIn.addStat(StatList.getObjectsPickedUpStats(item), i);
			}
		}
	}

	/**
	 * Get the name of this object. For players this returns their username
	 */
	public String getName() {
		return this.hasCustomName() ? this.getCustomNameTag()
				: I18n.translateToLocal("item." + this.getEntityItem().getUnlocalizedName());
	}

	/**
	 * Returns true if it's possible to attack this entity with an item.
	 */
	public boolean canBeAttackedWithItem() {
		return false;
	}

	@Nullable
	public Entity changeDimension(int dimensionIn) {
		Entity entity = super.changeDimension(dimensionIn);

		if (!this.world.isRemote && entity instanceof EntityItem) {
			((EntityItem) entity).searchForOtherItemsNearby();
		}

		return entity;
	}

	/**
	 * Returns the ItemStack corresponding to the Entity (Note: if no item exists,
	 * will log an error but still return an ItemStack containing Block.stone)
	 */
	public ItemStack getEntityItem() {
		return (ItemStack) this.getDataManager().get(ITEM);
	}

	/**
	 * Sets the ItemStack for this entity
	 */
	public void setEntityItemStack(ItemStack stack) {
		this.getDataManager().set(ITEM, stack);
		this.getDataManager().setDirty(ITEM);
	}

	public String getOwner() {
		return this.owner;
	}

	public void setOwner(String owner) {
		this.owner = owner;
	}

	public String getThrower() {
		return this.thrower;
	}

	public void setThrower(String thrower) {
		this.thrower = thrower;
	}

	public int getAge() {
		return this.age;
	}

	public void setDefaultPickupDelay() {
		this.delayBeforeCanPickup = 10;
	}

	public void setNoPickupDelay() {
		this.delayBeforeCanPickup = 0;
	}

	public void setInfinitePickupDelay() {
		this.delayBeforeCanPickup = 32767;
	}

	public void setPickupDelay(int ticks) {
		this.delayBeforeCanPickup = ticks;
	}

	public boolean cannotPickup() {
		return this.delayBeforeCanPickup > 0;
	}

	public void setNoDespawn() {
		this.age = -6000;
	}

	public void makeFakeItem() {
		this.setInfinitePickupDelay();
		this.age = 5999;
	}
}
